/*
 * Copyright (c) 2024 Panasonic Connect Co., Ltd.
 * 2024-06-28 Modified for Cameleo Project.
 */

#include "RtspToTS.hh"
#include "RtspToTSImpl.hh"
#include "RTSPCommon.hh"

#define RTSP_CLIENT_VERBOSITY_LEVEL 0 // set to 1 for more verbose output from the "RTSPClient"
#define RTSP_CLIENT_PROGRAMM_NAME NULL // no special meaning, NULL is OK
#define RTSP_CLIENT_TUNNEL_OVER_HTTP_PORT_NUM 0

#define RUN_THREAD 0
#define STOP_THREAD 1

portNumBits tunnelOverHTTPPortNum = 0;


////////// RtspToTS //////////

RtspToTS::RtspToTS() : pimpl(std::make_unique<Impl>()) {}

RtspToTS::RtspToTS(const char* rtspURL,
			const char* username,
			const char* password,
			unsigned int& outputFileDuration,
			const char* outputFilePath,
			const char* outputFileName,
			unsigned int& maxOutPutFileNum,
			Boolean isOutPutAudio,
			unsigned int& keepAliveInterval)
	: pimpl(std::make_unique<Impl>(rtspURL, username, password, outputFileDuration,
		outputFilePath, outputFileName, maxOutPutFileNum, isOutPutAudio, keepAliveInterval)) {}

RtspToTS::~RtspToTS() {}		

Boolean RtspToTS::setRtspURL(const char* rtspURL) {
	return pimpl->setRtspURL(rtspURL);
}

Boolean RtspToTS::setUserNameAndPassWord(const char* username, const char* password) {
	return pimpl->setUserNameAndPassWord(username, password);
}

Boolean RtspToTS::setOutPutFileDuration(unsigned int outputFileDuration) {
	return pimpl->setOutPutFileDuration(outputFileDuration);
}

Boolean RtspToTS::setOutPutFilePath(const char* outputFilePath) {
	return pimpl->setOutPutFilePath(outputFilePath);
}

Boolean RtspToTS::setOutPutFileName(const char* outputFileName) {
	return pimpl->setOutPutFileName(outputFileName);
}

Boolean RtspToTS::setMaxOutPutFileNum(unsigned int maxOutPutFileNum) {
	return pimpl->setMaxOutPutFileNum(maxOutPutFileNum);
}

Boolean RtspToTS::setIsOutPutAudio(Boolean isOutPutAudio) {
	return pimpl->setIsOutPutAudio(isOutPutAudio);
}

Boolean RtspToTS::setKeepAliveInterval(unsigned int keepAliveInterval) {
	return pimpl->setKeepAliveInterval(keepAliveInterval);
}

Boolean RtspToTS::setResultCallbackFunc(ResultCallbackFunc* callbackFunc) {
	return pimpl->setResultCallbackFunc(callbackFunc);
}

Boolean RtspToTS::setLogCallbackFunc(LogCallbackFunc* callbackFunc) {
	return pimpl->setLogCallbackFunc(callbackFunc);
}

const char* RtspToTS::getRtspURL() {
	return pimpl->getRtspURL();
}

const char* RtspToTS::getUserName() {
	return pimpl->getUserName();
}

const char* RtspToTS::getPassWord() {
	return pimpl->getPassWord();
}

unsigned int RtspToTS::getOutPutFileDuration() {
	return pimpl->getOutPutFileDuration();
}

const char* RtspToTS::getOutPutFilePath() {
	return pimpl->getOutPutFilePath();
}

const char* RtspToTS::getOutPutFileName() {
	return pimpl->getOutPutFileName();
}

unsigned int RtspToTS::getMaxOutPutFileNum() {
	return pimpl->getMaxOutPutFileNum();
}

Boolean RtspToTS::getIsOutPutAudio() {
	return pimpl->getIsOutPutAudio();
}

unsigned int RtspToTS::getKeepAliveInterval() {
	return pimpl->getKeepAliveInterval();
}

Boolean RtspToTS::startOutPut() {
	return pimpl->startOutPut();
}

Boolean RtspToTS::stopOutPut() {
	return pimpl->stopOutPut();
}

ThreadStatus RtspToTS::getThreadStatus() {
	return pimpl->getThreadStatus();
}


////////// Impl //////////

RtspToTS::Impl::Impl() {
	fRtspURL = strDup("");
	fUserName = strDup("");
	fPassWord = strDup("");
	fOutPutFileDuration = 1;
	fOutPutFilePath = strDup("");
	fOutPutFileName = strDup("RtspToTS");
	fMaxOutPutFileNum = 20;
	fIsOutPutAudio = False;
	fKeepAliveInterval = 55;
	
	initializeForThread();
}

RtspToTS::Impl::Impl(const char* rtspURL,
			const char* username,
			const char* password,
			unsigned int& outputFileDuration,
			const char* outputFilePath,
			const char* outputFileName,
			unsigned int& maxOutPutFileNum,
			Boolean isOutPutAudio,
			unsigned int& keepAliveInterval) {
	fRtspURL = strDup(rtspURL);
	fUserName = strDup(username);
	fPassWord = strDup(password);

	if(checkOutPutFileDuration(outputFileDuration)) {
		fOutPutFileDuration = outputFileDuration;
	}
	else {
		fOutPutFileDuration = outputFileDuration = 0;
	}
	
	fOutPutFilePath = correctiveOutPutFilePath(outputFilePath);
	fOutPutFileName = strDup(outputFileName);

	if(checkMaxOutPutFileNum(maxOutPutFileNum)) {
		fMaxOutPutFileNum = maxOutPutFileNum;
	}
	else {
		fMaxOutPutFileNum = maxOutPutFileNum = 0;
	}

	fIsOutPutAudio = isOutPutAudio;

	if(checkKeepAliveInterval(keepAliveInterval)) {
		fKeepAliveInterval = keepAliveInterval;
	}
	else {
		fKeepAliveInterval = keepAliveInterval = 55;
	}
	
	initializeForThread();
}

RtspToTS::Impl::~Impl() {
	stopOutPut();
	
	delete fWatchVariable;	

	delete[] fRtspURL;
	delete[] fUserName;
	delete[] fPassWord;
	delete[] fOutPutFilePath;
	delete[] fOutPutFileName;		
}

const char* RtspToTS::Impl::correctiveOutPutFilePath(const char* outputFilePath) {
	int outputFilePathLength = strlen(outputFilePath);
	
	if(outputFilePathLength != 0) {	
		if(outputFilePath[outputFilePathLength-1] != '/' ) {
			char tmpOutputFilePath[outputFilePathLength+2] = {0};
			sprintf(tmpOutputFilePath, "%s/", outputFilePath);			
			return strDup(tmpOutputFilePath);
		}
	}

	return strDup(outputFilePath);
}

Boolean RtspToTS::Impl::checkOutPutFileDuration(unsigned int outputFileDuration) {
	if(outputFileDuration >= 1 && outputFileDuration <= 3600) {
		return True;
	}
	return False;
}

Boolean RtspToTS::Impl::checkMaxOutPutFileNum(unsigned int maxOutPutFileNum) {
	if(maxOutPutFileNum >= 1 && maxOutPutFileNum <= 3600) {
		return True;
	}
	return False;
}

Boolean RtspToTS::Impl::checkKeepAliveInterval(unsigned int keepAliveInterval) {
	if(keepAliveInterval >= 1 && keepAliveInterval <= 3600) {
		return True;
	}
	return False;
}

void RtspToTS::Impl::initializeForThread() {
	fMaxOutPutFileDuration = 0;
	fOutPutFileNum = 0;

	fWatchVariable = new char(STOP_THREAD);
	fAuthenticator = NULL;
	
	fThread = NULL;
	fResultCallbackFunc = NULL;
	fLogCallbackFunc = NULL;
		
	fScheduler = NULL;
	fEnv = NULL;	
	fRtspClient = NULL;
	fServerSupportsGetParameter = False;
	fTransportStream = NULL;
	fSession = NULL;	
	fIter = NULL;
	fSubsession = NULL;
	fNumUsableSubsessions = 0;
	fHasUsableVideoSource = False;
	fSink = NULL;
	fStreamDuration = 0.0;
	fHeartBeatCommandTask = NULL;
	fHead = NULL;
	fTail = NULL;
	fRemoveTSFileNameWithPath = NULL;
	fM3U8FileNameWithPath = NULL;
	//fIsFirstTimeOutPutM3U8 = False;
	fTotalDuration = 0.0;
}

Boolean RtspToTS::Impl::setRtspURL(const char* rtspURL) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		delete[] fRtspURL;
		fRtspURL = strDup(rtspURL);
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setRtspURL\n");
	return False;
}

Boolean RtspToTS::Impl::setUserNameAndPassWord(const char* username, const char* password) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		delete[] fUserName;
		delete[] fPassWord;
		fUserName = strDup(username);
		fPassWord = strDup(password);
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setUserNameAndPassWord\n");
	return False;
}

Boolean RtspToTS::Impl::setOutPutFileDuration(unsigned int outputFileDuration) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		if(checkOutPutFileDuration(outputFileDuration)) {
			fOutPutFileDuration = outputFileDuration;
			return True;
		}
		else {
			UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "OutputFileDuration is out of range. Failed to setOutPutFileDuration.\n");
			return False;
		}
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setOutPutFileDuration\n");
	return False;
}

Boolean RtspToTS::Impl::setOutPutFilePath(const char* outputFilePath) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		delete[] fOutPutFilePath;
		fOutPutFilePath = correctiveOutPutFilePath(outputFilePath);
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setOutPutFilePath\n");
	return False;
}

Boolean RtspToTS::Impl::setOutPutFileName(const char* outputFileName) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		delete[] fOutPutFileName;
		fOutPutFileName = strDup(outputFileName);
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setOutPutFileName\n");
	return False;
}

Boolean RtspToTS::Impl::setMaxOutPutFileNum(unsigned int maxOutPutFileNum) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		if(checkMaxOutPutFileNum(maxOutPutFileNum)) {
			fMaxOutPutFileNum = maxOutPutFileNum;
			return True;
		}
		else {
			UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "MaxOutPutFileNum is out of range. Failed to setMaxOutPutFileNum.\n");
			return False;
		}
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setMaxOutPutFileNum\n");
	return False;
}

Boolean RtspToTS::Impl::setIsOutPutAudio(Boolean isOutPutAudio) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		fIsOutPutAudio = isOutPutAudio;
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setIsOutPutAudio\n");
	return False;
}

Boolean RtspToTS::Impl::setKeepAliveInterval(unsigned int keepAliveInterval) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		if(checkKeepAliveInterval(keepAliveInterval)) {
			fKeepAliveInterval = keepAliveInterval;
			return True;
		}
		else {
			UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "KeepAliveInterval is out of range. Failed to setKeepAliveInterval.\n");
			return False;
		}
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setKeepAliveInterval\n");
	return False;
}

Boolean RtspToTS::Impl::setResultCallbackFunc(ResultCallbackFunc* callbackFunc) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		fResultCallbackFunc = callbackFunc;
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setResultCallbackFunc\n");
	return False;
}

Boolean RtspToTS::Impl::setLogCallbackFunc(LogCallbackFunc* callbackFunc) {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus != THREAD_RUNNING) {
		//fLogCallbackFunc = callbackFunc;
		UsageEnvironment::setLogCallbackFunc(callbackFunc);
		return True;
	}

	UsageEnvironment::printLog(WARNING_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is running.Failed to setLogCallbackFunc\n");
	return False;
}

const char* RtspToTS::Impl::getRtspURL() {
	return fRtspURL;
}

const char* RtspToTS::Impl::getUserName() {
	return fUserName;
}

const char* RtspToTS::Impl::getPassWord() {
	return fPassWord;
}

unsigned int RtspToTS::Impl::getOutPutFileDuration() {
	return fOutPutFileDuration;
}

const char* RtspToTS::Impl::getOutPutFilePath() {
	return fOutPutFilePath;
}

const char* RtspToTS::Impl::getOutPutFileName() {
	return fOutPutFileName;
}

unsigned int RtspToTS::Impl::getMaxOutPutFileNum() {
	return fMaxOutPutFileNum;
}

Boolean RtspToTS::Impl::getIsOutPutAudio() {
	return fIsOutPutAudio;
}

unsigned int RtspToTS::Impl::getKeepAliveInterval() {
	return fKeepAliveInterval;
}

Boolean RtspToTS::Impl::startOutPut() {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus == THREAD_RUNNING) {
		return True;			
	}
	else if(threadStatus == THREAD_ERROR) {
		if(!cleanThread()) {
			UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to cleanThread\n");
			return False;
		}		
	}

	if(fOutPutFileDuration == 0 || fMaxOutPutFileNum == 0 ) {
		UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "OutputFileDuration or maxOutPutFileNum is out of range. Failed to startOutPut\n");
		return False;
	}

	fThread = new std::thread(&RtspToTS::Impl::realRunProcess, this);
	if(fThread->joinable()) {
		UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Start output\n");
		return True;
	}
	else {
		UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to create thread\n");
		return False;
	}
	return True;
}

Boolean RtspToTS::Impl::stopOutPut() {
	ThreadStatus threadStatus = getThreadStatus();
	if(threadStatus == THREAD_STOPPED) {
		return True;
	}

	Boolean result = cleanThread();

	UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Stop output\n");

	return result;
}

ThreadStatus RtspToTS::Impl::getThreadStatus() {
	if(*fWatchVariable == STOP_THREAD) {
		if(fThread != NULL) {
			return THREAD_ERROR;
		}
		else {
			return THREAD_STOPPED;
		}
	}
	else {
		if(fThread != NULL) {
			return THREAD_RUNNING;
		}
		else {
			return THREAD_ERROR;
		}
	}
}


void RtspToTS::Impl::realRunProcess() {	
	
	*fWatchVariable = RUN_THREAD;
	fAuthenticator = new Authenticator(fUserName, fPassWord);
	
	fScheduler = BasicTaskScheduler::createNew();
	fEnv = BasicUsageEnvironment::createNew(*fScheduler);
	
	fRtspClient = RTSPClient::createNew(*fEnv, fRtspURL, RTSP_CLIENT_VERBOSITY_LEVEL, RTSP_CLIENT_PROGRAMM_NAME, tunnelOverHTTPPortNum);
    if (fRtspClient == NULL) {
      //*fEnv << "Failed to create a RTSP client for URL \"" << fRtspURL << "\": " << fEnv->getResultMsg() << "\n";
      UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to create a RTSP client for URL \"%s\": %s\n", fRtspURL, fEnv->getResultMsg());
	  
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
      terminateThread();
	  return;
    }

	fRtspClient->sendOptionsCommand(&RtspToTS::Impl::continueAfterOPTIONS, this, fAuthenticator);

	// All further processing will be done from within the event loop:	
  	fEnv->taskScheduler().doEventLoop(fWatchVariable); // does not return
}

// A function that outputs a string that identifies each stream (for debugging output).
UsageEnvironment& operator<<(UsageEnvironment& env, const RTSPClient& rtspClient) {
  return env << "[URL:\"" << rtspClient.url() << "\"]: ";
}

// A function that outputs a string that identifies each subsession (for debugging output).
UsageEnvironment& operator<<(UsageEnvironment& env, const MediaSubsession& subsession) {
  return env << subsession.mediumName() << "/" << subsession.codecName();
}

void RtspToTS::Impl::continueAfterOPTIONS(void* clientData, int resultCode, char* resultString) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->continueAfterOPTIONS1(resultCode, resultString);
	}	
}

void RtspToTS::Impl::continueAfterOPTIONS1(int resultCode, char* resultString) {
	if (resultCode == 0) {
		// Note whether the server told us that it supports the "GET_PARAMETER" command:
		fServerSupportsGetParameter = RTSPOptionIsSupported("GET_PARAMETER", resultString);		
	}
	delete[] resultString;

	fRtspClient->sendDescribeCommand(&RtspToTS::Impl::continueAfterDESCRIBE, this, fAuthenticator);
}


void RtspToTS::Impl::continueAfterDESCRIBE(void* clientData, int resultCode, char* resultString) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->continueAfterDESCRIBE1(resultCode, resultString);
	}	
}

void RtspToTS::Impl::continueAfterDESCRIBE1(int resultCode, char* resultString) {
  do {
    if (resultCode != 0) {
      //*fEnv << *fRtspClient << "Failed to get a SDP description: " << resultString << "\n";
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Failed to get a SDP description: %s\n", fRtspClient->url(), resultString);
      delete[] resultString;
      break;
    }
	
    // Create a Transport Stream to multiplex the Elementary Stream data from each subsession:
    fTransportStream = MPEG2TransportStreamFromESSource::createNew(*fEnv);

    // Create a media session object from the SDP description.
    // Then iterate over it, to look for subsession(s) that we can handle:
    fSession = MediaSession::createNew(*fEnv, resultString);
    delete[] resultString;
	 
    if (fSession == NULL) {
      //*fEnv << *fRtspClient << "Failed to create a MediaSession object from the SDP description: " << fEnv->getResultMsg() << "\n";
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Failed to create a MediaSession object from the SDP description: %s\n", fRtspClient->url(), fEnv->getResultMsg());
      break;
    } else if (!fSession->hasSubsessions()) {
      //*fEnv << *fRtspClient << "This session has no media subsessions (i.e., no \"m=\" lines)\n";
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: This session has no media subsessions (i.e., no \"m=\" lines)\n", fRtspClient->url());
      break;
    }

    fIter = new MediaSubsessionIterator(*fSession);
    setupNextSubsession();
    return;
  } while (0);

  // An error occurred:
  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
  terminateThread();
}

void RtspToTS::Impl::setupNextSubsession() {
  fSubsession = fIter->next();
  if (fSubsession != NULL) {
    // Check whether this subsession is a codec that we support.
    // We support H.264 video, and AAC audio.
    if ((strcmp(fSubsession->mediumName(), "video") == 0 &&
	 strcmp(fSubsession->codecName(), "H264") == 0) ||
	(strcmp(fSubsession->mediumName(), "audio") == 0 &&
	 strcmp(fSubsession->codecName(), "MPEG4-GENERIC"/*aka. AAC*/) == 0 &&
	 fIsOutPutAudio)) {
      // Use this subsession.
      ++fNumUsableSubsessions;
      if (!fSubsession->initiate()) {
	//*fEnv << *fRtspClient << "Failed to initiate the \"" << *fSubsession << "\" subsession: " << fEnv->getResultMsg() << "\n";
	UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Failed to initiate the \"%s/%s\" subsession: %s\n", fRtspClient->url(), fSubsession->mediumName(), fSubsession->codecName(), fEnv->getResultMsg());
      } else {
	//*fEnv << *fRtspClient << "Initiated the \"" << *fSubsession << "\" subsession\n";
	UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Initiated the \"%s/%s\" subsession\n", fRtspClient->url(), fSubsession->mediumName(), fSubsession->codecName());
    
	// Continue setting up this subsession, by sending a RTSP "SETUP" command:
	fRtspClient->sendSetupCommand(*fSubsession, &RtspToTS::Impl::continueAfterSETUP, this, False, False,
				     False, fAuthenticator);
	return;
      }
    }      
    setupNextSubsession(); // give up on this subsession; go to the next one
    return;
  }

  // We've gone through all of the subsessions.
  if (fNumUsableSubsessions == 0) {
    //*fEnv << *fRtspClient << "This stream has no usable subsessions\n";
	UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: This stream has no usable subsessions\n", fRtspClient->url());
	
	UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
    fEnv->taskScheduler().stopEventLoop();
    return;
  }
  
  // We've gone through all of the subsessions.
  if (fHasUsableVideoSource == False) {
    //*fEnv << *fRtspClient << "This stream has no usable video source\n";
	UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: This stream has no usable video source\n", fRtspClient->url());
	
	UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
    fEnv->taskScheduler().stopEventLoop();
    return;
  }

  startPlayingSession();
}

void RtspToTS::Impl::continueAfterSETUP(void* clientData, int resultCode, char* resultString) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->continueAfterSETUP1(resultCode, resultString);	
	}
}

void RtspToTS::Impl::continueAfterSETUP1(int resultCode, char* resultString) {
  do {
    if (resultCode != 0) {
      //*fEnv << *fRtspClient << "Failed to set up the \"" << *fSubsession << "\" subsession: " << resultString << "\n";
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Failed to set up the \"%s/%s\" subsession\n", fRtspClient->url(), fSubsession->mediumName(), fSubsession->codecName());
      break;
    }
    delete[] resultString;

   // *fEnv << *fRtspClient << "Set up the \"" << *fSubsession << "\" subsession\n";
	UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Set up the \"%s/%s\" subsession\n", fRtspClient->url(), fSubsession->mediumName(), fSubsession->codecName());

    // Feed this subsession's input source into the Transport Stream:
    if (strcmp(fSubsession->mediumName(), "video") == 0) {
      // Create a 'framer' filter for the input source, to put the stream of NAL units into a
      // form that's usable to output to the Transport Stream.
      // (Note that we use a *DiscreteFramer*, because the input source is a stream of discrete
      //  NAL units - i.e., one at a time.)
      H264or5VideoStreamDiscreteFramer* framer;
      int mpegVersion;

      if (strcmp(fSubsession->codecName(), "H264") == 0) {
	mpegVersion = 5; // for H.264
	framer = H264VideoStreamDiscreteFramer::createNew(*fEnv, fSubsession->readSource(),
							  True/*includeStartCodeInOutput*/,
							  True/*insertAccessUnitDelimiters*/);

	// Add any known SPS and PPS NAL units to the framer, so they'll get output ASAP:
	u_int8_t* sps = NULL; unsigned spsSize = 0;
	u_int8_t* pps = NULL; unsigned ppsSize = 0;
	unsigned numSPropRecords;
	SPropRecord* sPropRecords
	  = parseSPropParameterSets(fSubsession->fmtp_spropparametersets(), numSPropRecords);
	if (numSPropRecords > 0) {
	  sps = sPropRecords[0].sPropBytes;
	  spsSize = sPropRecords[0].sPropLength;
	}
	if (numSPropRecords > 1) {
	  pps = sPropRecords[1].sPropBytes;
	  ppsSize = sPropRecords[1].sPropLength;
	}
	framer->setVPSandSPSandPPS(NULL, 0, sps, spsSize, pps, ppsSize);
	delete[] sPropRecords;
      } else { // H.265
	mpegVersion = 6; // for H.265
	framer = H265VideoStreamDiscreteFramer::createNew(*fEnv, fSubsession->readSource(),
							  True/*includeStartCodeInOutput*/,
							  True/*insertAccessUnitDelimiters*/);
	
	// Add any known VPS, SPS and PPS NAL units to the framer, so they'll get output ASAP:
	u_int8_t* vps = NULL; unsigned vpsSize = 0;
	u_int8_t* sps = NULL; unsigned spsSize = 0;
	u_int8_t* pps = NULL; unsigned ppsSize = 0;
	unsigned numSPropRecords;
	SPropRecord *sPropRecordsVPS, *sPropRecordsSPS, *sPropRecordsPPS;
	
	sPropRecordsVPS = parseSPropParameterSets(fSubsession->fmtp_spropvps(), numSPropRecords);
	if (numSPropRecords > 0) {
	  vps = sPropRecordsVPS[0].sPropBytes;
	  vpsSize = sPropRecordsVPS[0].sPropLength;
	}
	
	sPropRecordsSPS = parseSPropParameterSets(fSubsession->fmtp_spropsps(), numSPropRecords);
	if (numSPropRecords > 0) {
	  sps = sPropRecordsSPS[0].sPropBytes;
	  spsSize = sPropRecordsSPS[0].sPropLength;
	}
	
	sPropRecordsPPS = parseSPropParameterSets(fSubsession->fmtp_sproppps(), numSPropRecords);
	if (numSPropRecords > 0) {
	  pps = sPropRecordsPPS[0].sPropBytes;
	  ppsSize = sPropRecordsPPS[0].sPropLength;
	}
	
	framer->setVPSandSPSandPPS(vps, vpsSize, sps, spsSize, pps, ppsSize);
	delete[] sPropRecordsVPS; delete[] sPropRecordsSPS; delete[] sPropRecordsPPS;
      }
      
      fTransportStream->addNewVideoSource(framer, mpegVersion);
	  fHasUsableVideoSource = True;
    } else { // audio (AAC)
      // Create a 'framer' filter for the input source, to add a ADTS header to each AAC frame,
      // to make the audio playable.
      ADTSAudioStreamDiscreteFramer* framer
	= ADTSAudioStreamDiscreteFramer::createNew(*fEnv, fSubsession->readSource(),
						   fSubsession->fmtp_config());
      fTransportStream->addNewAudioSource(framer, 4/*mpegVersion: AAC*/);
    }

    // Also set up BYE handler//#####@@@@@
  } while (0);

  // Set up the next subsession, if any:
  setupNextSubsession();
}

void RtspToTS::Impl::startPlayingSession() {
  // Having successfully setup the session, create a data sink for it, and call "startPlaying()" on it.
  // (This will prepare the data sink to receive data; the actual flow of data from the client won't start happening until later,
  // after we've sent a RTSP "PLAY" command.)

  fSink = HLSSegmenter::createNew(*fEnv, fOutPutFileDuration, fOutPutFilePath, fOutPutFileName, segmentationCallback, this);

  // Start playing the sink object:
  //*fEnv << "Beginning to read...\n";
  UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Beginning to read...\n");
  
  fSink->startPlaying(*fTransportStream, &RtspToTS::Impl::afterPlaying, this);

  // Now, send a RTSP "PLAY" command to start the streaming:
  if (fSession->absStartTime() != NULL) {
    // Special case: The stream is indexed by 'absolute' time, so send an appropriate "PLAY" command:
    fRtspClient->sendPlayCommand(*fSession, continueAfterPLAY, this, fSession->absStartTime(), fSession->absEndTime(), 1.0f, fAuthenticator);
  } else {
    fStreamDuration = fSession->playEndTime() - fSession->playStartTime();
    fRtspClient->sendPlayCommand(*fSession, continueAfterPLAY, this, 0.0f, -1.0f, 1.0f, fAuthenticator);
  }
}

void RtspToTS::Impl::segmentationCallback(void* clientData, const char * segmentFileName, double segmentDuration) {
	if(clientData != NULL){
		((RtspToTS::Impl*)clientData)->segmentationCallback1(segmentFileName, segmentDuration);
	}	
}

void RtspToTS::Impl::segmentationCallback1(const char * segmentFileName, double segmentDuration) {
	// Notify ts file information
	if(fResultCallbackFunc != NULL) {
		(*fResultCallbackFunc)(segmentFileName, segmentDuration);
	}
	
	++fOutPutFileNum;

	if(fMaxOutPutFileDuration < segmentDuration) {
		fMaxOutPutFileDuration = segmentDuration + 1;
	}
		
	// Update our list of segments:
	SegmentRecord* newSegment = new SegmentRecord(segmentFileName, segmentDuration);
	if (fTail != NULL) {
		fTail->next() = newSegment;
	} else {
		fHead = newSegment;
	}
	fTail = newSegment;
	fTotalDuration += segmentDuration;

	//*fEnv <<"Wrote segment \""<<segmentFileName<<"\" (duration: "<<segmentDuration<<" seconds) -> "<<fTotalDuration<<" seconds of data stored\n";
	UsageEnvironment::printLog(DEBUG_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Wrote segment \"%s\" (duration: %f seconds) -> %f seconds of data stored\n", segmentFileName, segmentDuration, fTotalDuration);

	static unsigned firstSegmentCounter = 1;
	while (fOutPutFileNum > fMaxOutPutFileNum) {
		// Remove segments from the head of the list:
		SegmentRecord* segmentToRemove = fHead;
		if (segmentToRemove == NULL) { // should not happen
		  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
		  fEnv->taskScheduler().stopEventLoop();
		  return;
		}

		fHead = segmentToRemove->next();
		if (fTail == segmentToRemove) { // should not happen
		  fTail = NULL;
		}
		segmentToRemove->next() = NULL;

		fTotalDuration -= segmentToRemove->duration();
		//*fEnv <<"\tDeleting segment \""<<segmentToRemove->fileName()<<"\" (duration: "<<segmentToRemove->duration()<<" seconds) -> "<<fTotalDuration<<" seconds of data stored\n";
		UsageEnvironment::printLog(DEBUG_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "\tDeleting segment \"%s\" (duration: %f seconds) -> %f seconds of data stored\n", segmentToRemove->fileName(), segmentToRemove->duration(), fTotalDuration);
		
		if (fRemoveTSFileNameWithPath == NULL) {
			fRemoveTSFileNameWithPath = new char[strlen(fOutPutFilePath) + strlen(segmentToRemove->fileName()) + 1];
			if (fRemoveTSFileNameWithPath == NULL) {
			  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to allocate memory.Thread is interrupted!!\n");
			  fEnv->taskScheduler().stopEventLoop();
			  return;
			}			
		}
		sprintf(fRemoveTSFileNameWithPath, "%s%s", fOutPutFilePath, segmentToRemove->fileName());
		
		if (unlink(fRemoveTSFileNameWithPath) != 0) {
		  //*fEnv << "\t\tunlink(\"" << fRemoveTSFileNameWithPath << "\") failed\n";
		  UsageEnvironment::printLog(DEBUG_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "\t\tunlink(\"%s\") failed\n", fRemoveTSFileNameWithPath);
		}
		delete segmentToRemove;
		++firstSegmentCounter;
		--fOutPutFileNum;
	}

	// Then, rewrite our ".m3u8" file with the new list of segments:
	if (fM3U8FileNameWithPath == NULL) {
		fM3U8FileNameWithPath = new char[strlen(fOutPutFilePath) + strlen(fOutPutFileName) + 5/*strlen(".m3u8")*/ + 1];
		if (fM3U8FileNameWithPath == NULL) {
		  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to allocate memory.Thread is interrupted!!\n");
		  fEnv->taskScheduler().stopEventLoop();
		  return;
		}
		sprintf(fM3U8FileNameWithPath, "%s%s.m3u8", fOutPutFilePath, fOutPutFileName);
	}

	// Open our ".m3u8" file for output, and write the prefix:
	FILE* ourM3U8Fid = fopen(fM3U8FileNameWithPath, "wb");
	if (ourM3U8Fid == NULL) {
		//*fEnv << "Failed to open file \"" << fM3U8FileNameWithPath << "\": " << fEnv->getResultMsg();
		UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Failed to open file \"%s\"\n", fM3U8FileNameWithPath);

		UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
		fEnv->taskScheduler().stopEventLoop();
		return;
	}

	fprintf(ourM3U8Fid,
	  "#EXTM3U\n"
	  "#EXT-X-VERSION:3\n"
	  "#EXT-X-INDEPENDENT-SEGMENTS\n"
	  "#EXT-X-TARGETDURATION:%u\n"
	  "#EXT-X-MEDIA-SEQUENCE:%u\n",
	  fMaxOutPutFileDuration,
	  firstSegmentCounter);

	// Write the list of segments:
	for (SegmentRecord* segment = fHead; segment != NULL; segment = segment->next()) {
		fprintf(ourM3U8Fid,
		    "#EXTINF:%f,\n"
		    "%s\n",
		    segment->duration(),
		    segment->fileName());
	}

	// Close our ".m3u8" file:
	fclose(ourM3U8Fid);

	static Boolean isFirstTime = True;
	if (isFirstTime) {
		//*fEnv <<"Wrote index file \""<<fOutPutFileName<<"\"; the stream can now be played from a URL pointing to this file.\n";
		UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Wrote index file \"%s\"; the stream can now be played from a URL pointing to this file.\n", fOutPutFileName);
		
		isFirstTime = False;
	}
}

void RtspToTS::Impl::afterPlaying(void* clientData) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->afterPlaying1();
	}
}

void RtspToTS::Impl::afterPlaying1() {
	//*fEnv << "...Done reading\n";
	UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "...Done reading.\n");
	
	fEnv->taskScheduler().stopEventLoop();
}

void RtspToTS::Impl::continueAfterPLAY(void* clientData, int resultCode, char* resultString) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->continueAfterPLAY1(resultCode, resultString);
	}
}

void RtspToTS::Impl::continueAfterPLAY1(int resultCode, char* resultString) {
  do {
    if (resultCode != 0) {
      //*fEnv << *fRtspClient << "Failed to start playing session: " << resultString << "\n";
	  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Failed to start playing session: %s\n", fRtspClient->url(), resultString);
      break;
    }
    delete[] resultString;

    // Set timer based on duration #####@@@@@

    //*fEnv << *fRtspClient << "Started playing session";
	UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "[URL:\"%s\"]: Started playing session\n", fRtspClient->url());
    if (fStreamDuration > 0) {
      //*fEnv << " (for up to " << fStreamDuration << " seconds)";
	  UsageEnvironment::printLog(INFO_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, " (for up to %f seconds)...\n", fStreamDuration);
    }
    //*fEnv << "...\n";

	scheduleHeartBeatCommand();

    return;
  } while (0);

  // An error occurred:
  UsageEnvironment::printLog(ERROR_LEVEL_LOG, __FILE__, __FUNCTION__, __LINE__, "Thread is interrupted!!\n");
  fEnv->taskScheduler().stopEventLoop();
}

void RtspToTS::Impl::scheduleHeartBeatCommand()
{
	unsigned secondsToDelay = (fRtspClient->sessionTimeoutParameter())/2;
	if(secondsToDelay <= 0) {
		secondsToDelay = fKeepAliveInterval;
	}
	
	fHeartBeatCommandTask = fEnv->taskScheduler().scheduleDelayedTask(secondsToDelay*1000000, sendHeartBeatCommand, this);
}

void RtspToTS::Impl::sendHeartBeatCommand(void* clientData) {
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->sendHeartBeatCommand1();
	}
}

void RtspToTS::Impl::sendHeartBeatCommand1() {
	if (fServerSupportsGetParameter)
	{
		fRtspClient->sendGetParameterCommand(*fSession, &RtspToTS::Impl::continueAfterHeartBeatOption, this, NULL, fAuthenticator);
	}
	else
	{
		fRtspClient->sendOptionsCommand(&RtspToTS::Impl::continueAfterHeartBeatOption, this, fAuthenticator);
	}
}

void RtspToTS::Impl::continueAfterHeartBeatOption(void* clientData, int resultCode, char* resultString) {	
	delete[] resultString;
	
	if(clientData != NULL) {
		((RtspToTS::Impl*)clientData)->scheduleHeartBeatCommand();
	}
}

Boolean RtspToTS::Impl::cleanThread() {

	*fWatchVariable = STOP_THREAD;
	
	if(fThread->joinable()) {
		fThread->join();
	}
	
	delete fThread;
	fThread = NULL;

	fEnv->taskScheduler().unscheduleDelayedTask(fHeartBeatCommandTask);
	fHeartBeatCommandTask = NULL;

	if(fRtspClient != NULL && fSession != NULL) {
		fRtspClient->sendTeardownCommand(*fSession, NULL, this, fAuthenticator);
	}

	delete fIter;
	fIter = NULL;
	
	Medium::close(fSink);
	fSink = NULL;

	Medium::close(fTransportStream);
	fTransportStream = NULL;

	Medium::close(fSession);
	fSession = NULL;	
	
	Medium::close(fRtspClient);
	fRtspClient = NULL;

	fEnv->reclaim();
	fEnv = NULL;
	
	delete fScheduler;
	fScheduler = NULL;

	delete fAuthenticator;
	fAuthenticator = NULL;

	delete fHead;
	fHead = NULL;
	fTail = NULL;

	delete[] fRemoveTSFileNameWithPath;
	fRemoveTSFileNameWithPath = NULL;
	
	delete[] fM3U8FileNameWithPath;
	fM3U8FileNameWithPath = NULL;
	
	fMaxOutPutFileDuration = 0;
	fOutPutFileNum = 0;
	fServerSupportsGetParameter = False;
	fSubsession = NULL;
	fNumUsableSubsessions = 0;
	fHasUsableVideoSource = False;
	fStreamDuration = 0.0;
	fTotalDuration = 0.0;

	return True;
}


void RtspToTS::Impl::terminateThread() {
	*fWatchVariable = STOP_THREAD;
}


