/*
 * Copyright (c) 2024 Panasonic Connect Co., Ltd.
 * 2024-06-28 Modified for Cameleo Project.
 */

#ifndef _RTSP_TO_TS_IMPL_HH
#define _RTSP_TO_TS_IMPL_HH

#ifndef _RTSP_TO_TS_HH
#include "RtspToTS.hh"
#endif

#ifndef _LIVEMEDIA_HH
#include "liveMedia.hh"
#endif

#ifndef _STRDUP_HH
#include "strDup.hh"
#endif

#include <thread>

class RtspToTS::Impl {
public:
	Impl();
	
	Impl(const char* rtspURL,
			const char* username,
			const char* password,
			unsigned int& outputFileDuration,
			const char* outputFilePath,
			const char* outputFileName,
			unsigned int& maxOutPutFileNum,
			Boolean isOutPutAudio,
			unsigned int& keepAliveInterval);

	~Impl();

	Boolean setRtspURL(const char* rtspURL);
	
	Boolean setUserNameAndPassWord(const char* username, const char* password);
	
	Boolean setOutPutFileDuration(unsigned int outputFileDuration);
	
	Boolean setOutPutFilePath(const char* outputFilePath);

	Boolean setOutPutFileName(const char* outputFileName);
	
	Boolean setMaxOutPutFileNum(unsigned int maxOutPutFileNum);
	
	Boolean setIsOutPutAudio(Boolean isOutPutAudio);

	Boolean setKeepAliveInterval(unsigned int keepAliveInterval);
	
	Boolean setResultCallbackFunc(ResultCallbackFunc* callbackFunc);

	Boolean setLogCallbackFunc(LogCallbackFunc* callbackFunc);
	
	const char* getRtspURL();
	
	const char* getUserName();
	
	const char* getPassWord();
	
	unsigned int getOutPutFileDuration();
	
	const char* getOutPutFilePath();

	const char* getOutPutFileName();
	
	unsigned int getMaxOutPutFileNum();
	
	Boolean getIsOutPutAudio();

	unsigned int getKeepAliveInterval();

	// start out put ts files
	Boolean startOutPut();
	
	// stop out put ts files
	Boolean stopOutPut();
	
	ThreadStatus getThreadStatus();
	
private:
	// if the output file path has errors, corrective it 
	const char* correctiveOutPutFilePath(const char* outputFilePath);

	// check whether output file duration is in the range
	Boolean checkOutPutFileDuration(unsigned int outputFileDuration);

	// check whether max output file num is in the range
	Boolean checkMaxOutPutFileNum(unsigned int maxOutPutFileNum);

	// check whether keep alive interval is in the range
	Boolean checkKeepAliveInterval(unsigned int keepAliveInterval);

	// initialize the variable used in thread
	void initializeForThread();	

	// perform actual RtspToTS function
	void realRunProcess();
	
	// clean the resource used in thread
	Boolean cleanThread();
	
	// when error occurs in thread, 
	void terminateThread();

	static void continueAfterOPTIONS(void* clientData, int resultCode, char* resultString);

	void continueAfterOPTIONS1(int resultCode, char* resultString);

	static void continueAfterDESCRIBE(void* clientData, int resultCode, char* resultString);

	void continueAfterDESCRIBE1(int resultCode, char* resultString);

	void setupNextSubsession();

	static void continueAfterSETUP(void* clientData, int resultCode, char* resultString);

	void continueAfterSETUP1(int resultCode, char* resultString);

	void startPlayingSession();

	static void segmentationCallback(void* clientData , const char * segmentFileName, double segmentDuration);

	void segmentationCallback1(const char * segmentFileName, double segmentDuration);

	static void afterPlaying(void* clientData);

	void afterPlaying1();

	static void continueAfterPLAY(void* clientData, int resultCode, char* resultString);

	void continueAfterPLAY1(int resultCode, char* resultString);

	void scheduleHeartBeatCommand();

	static void sendHeartBeatCommand(void* clientData);

	void sendHeartBeatCommand1();

	static void continueAfterHeartBeatOption(void* clientData, int resultCode, char* resultString);

private:
	const char* fRtspURL;
	const char* fUserName;
	const char* fPassWord;
	unsigned int fMaxOutPutFileDuration;	// max duration of output ts files
	unsigned int fOutPutFileDuration;	// ts file duration
	const char* fOutPutFilePath;
	const char* fOutPutFileName;	
	unsigned int fMaxOutPutFileNum;	// max number of ts files that can be saved
	unsigned int fOutPutFileNum;	// number of ts files saved now
	Boolean fIsOutPutAudio;	// output audio or not
	unsigned int fKeepAliveInterval;	// Time interval for sending keep alive(GET_PAREMETER) packet

	char volatile* fWatchVariable;	// control whether the thread continues to execute
	Authenticator* fAuthenticator;
	
	std::thread* fThread;
	ResultCallbackFunc* fResultCallbackFunc;
	LogCallbackFunc* fLogCallbackFunc;
		
	TaskScheduler* fScheduler;
	UsageEnvironment* fEnv;	
	RTSPClient* fRtspClient;
	Boolean fServerSupportsGetParameter;
	MPEG2TransportStreamFromESSource* fTransportStream;
	MediaSession* fSession;
	MediaSubsessionIterator* fIter;
	MediaSubsession* fSubsession;	
	unsigned int fNumUsableSubsessions;
	Boolean fHasUsableVideoSource;
	MediaSink* fSink;
	double fStreamDuration;
	TaskToken fHeartBeatCommandTask;
	SegmentRecord* fHead;
	SegmentRecord* fTail;
	char* fRemoveTSFileNameWithPath;
	char* fM3U8FileNameWithPath;	
	double fTotalDuration;
	
};

class RtspToTS::SegmentRecord {
public:
  SegmentRecord(const char * segmentFileName, double segmentDuration)
    : fNext(NULL), fSegmentFileName(strDup(segmentFileName)), fSegmentDuration(segmentDuration) {
  }
  virtual ~SegmentRecord() {
    delete[] fSegmentFileName;
    delete fNext;
  }

  SegmentRecord*& next() { return fNext; }
  const char * fileName() const { return fSegmentFileName; }
  double duration() const { return fSegmentDuration; }

private:
  SegmentRecord* fNext;
  char* fSegmentFileName;
  double fSegmentDuration;
};

#endif


